<?php
if (!defined('ABSPATH')) {
  exit; // Exit if accessed directly.
}

/**
 *
 * Template data shape:
 * [
 * 	name:"string",
 * 	description?:"string",
 * 	key?:"string"
 * 	icon_url?:"string"
 * 	is_active?:"bool"
 * 	is_featured?:"bool"
 * 	input_fields:[
 * 	 [field_type:"text|textarea|number|language|number_of_result|tone_of_voice|result_format", field_name:"string", field_placeholder?:"string", key:"string" ]	
 * 	 [field_type:"text|textarea|number|language|number_of_result|tone_of_voice|result_format", field_name:"string", field_placeholder?:"string", key:"string" ]	
 * 	 [field_type:"text|textarea|number|language|number_of_result|tone_of_voice|result_format", field_name:"string", field_placeholder?:"string", key:"string" ]	
 * 	]
 *	prompt:"string"
 * ]
 *
 */

class Ultimate_AI_Template_Manager
{
  private $cpt_type = "uai_cpt_template";
  private $meta_key = "template";
  private $blog_prompt_key = "uai_blog_templates";

  function __construct()
  {
  }

  function save_blog_template($payload)
  {
    return update_option($this->blog_prompt_key, $payload, false);
  }

  function get_blog_templates() //full_blog|title|outlines|regen|image
  {
    $prompts = get_option($this->blog_prompt_key, "");

    if (is_array($prompts) && !empty($prompts)) {
      return $prompts;
    }

    return Ultimate_AI_Default_Template_Manager::get_blog_templates_prompt("", false); //return full prompts array
  }

  function get_blog_template_prompt($type) //full_blog|title|outlines|regen|image
  {
    $prompts = get_option($this->blog_prompt_key, "");

    if (is_array($prompts) && isset($prompts[$type]) && !empty($prompts[$type])) {
      return $prompts[$type];
    }

    return Ultimate_AI_Default_Template_Manager::get_blog_templates_prompt($type);
  }

  //`prompt`:string
  //`var_key_value`: Array. key is the niddle to find and value is the value to replace 
  function replace_key_with_value($prompt, $var_key_value)
  {
    if (!is_array($var_key_value) || empty($var_key_value)) return $prompt;

    foreach ($var_key_value as $key => $value) {
      if (!empty($value)) {
        $prompt = str_replace('[[' . $key . ']]', $value, $prompt);
      }
    }

    return $prompt;
  }

  function blog_prompt_variable_keys()
  {
    return [
      "tone_of_voice" => "",
      "language" => "",
      "about" => "",
      "primary_key" => "",
      "title" => "",
      "outlines" => "",
      "secondary_key" => "",
      "length" => "",
      "image_prompt" => "",
      "regeneration_text" => "",
    ];
  }


  function ultimate_ai_create_template($name, $input_fields, $prompt, $desc = "", $key = "", $icon_url = "", $is_featured = false, $id = 0)
  {
    $template_data = $this->ultimate_ai_get_template_data_boilerplate();
    $template_data["name"] = $name;
    $template_data["input_fields"] = $input_fields;
    $template_data["prompt"] = $prompt;

    if (!empty($desc)) $template_data["description"] = $desc;
    if (!empty($key)) $template_data["key"] = $key;
    if (!empty($icon_url)) $template_data["icon_url"] = $icon_url;
    if (!empty($is_featured)) $template_data["is_featured"] = $is_featured;

    $user = wp_get_current_user();

    $cpt_args = [
      "ID"       => $id,
      "post_type"     => $this->cpt_type,
      "post_author"  => $user->ID,
      "post_content" => "",
      "post_title"   => $name,
      "post_status"  => "publish",
      "meta_input"   => array(
        "template" => $template_data,
        "is_featured" => $template_data["is_featured"],
        "is_active" => $template_data["is_active"],
      ),
    ];

    $res = wp_insert_post($cpt_args);

    if (is_wp_error($res) || $res == 0) {
      return ["is_success" => false, "message" => $res->get_error_message() ?? __("unknown message", ULTIMATE_AI_SLUG)];
    }

    $template_data["id"] = $res;

    return ["is_success" => true, "data" => $template_data];
  }

  function ultimate_ai_delete_template($template_id)
  {
    $is_success = wp_delete_post($template_id);
    if (!$is_success) return ["is_success" => false, "message" => __("Template deleation fail", ULTIMATE_AI_SLUG)];
    return ["is_success" => true, "data" => "Template deleted"];
  }

  function ultimate_ai_update_template_meta($key, $value, $id, $is_direct_meta = false)
  {
    $template = get_post_meta($id, $this->meta_key, true);
    if (!$template) return ["is_success" => false, "message" => __("Template meta not found", ULTIMATE_AI_SLUG)];
    $template[$key] = $value;
    $is_success = update_post_meta($id, $this->meta_key, $template);
    if ($is_direct_meta) {
      $is_success = update_post_meta($id, $key, $value);
    }
    if (!$is_success) return ["is_success" => false, "message" => __("Template update fail", ULTIMATE_AI_SLUG)];
    return ["is_success" => true, "data" => $template];
  }

  function ultimate_ai_get_templates($filter_array = [])
  {
    $args = [
      "post_type" => $this->cpt_type,
      "post_status" => "publish",
      "posts_per_page" => -1, // all
      'fields' => 'ids'
    ];

    $args = array_merge($filter_array, $args);

    $templates = [];

    $templates = [];
    $ids = get_posts($args);

    foreach ($ids as $id) {
      $meta = get_post_meta($id, $this->meta_key, true);
      $meta["date"] = get_post_time('U', false, $id, false);
      $meta["id"] = $id;
      $templates[] = $meta;
    }

    return ["templates" => $templates, "total_templates" => count($ids)];
  }


  function ultimate_ai_get_template_by_id($template_id)
  {
    $post = get_post($template_id);
    if ($post == null) return false;

    $template = get_post_meta($template_id, "template", true);
    $template["date"] = get_post_time('U', false, $template_id, false);
    $template["id"] = $template_id;
    return $template;
  }

  function ultimate_ai_template_name_exist_by($key, $value)
  {
    $args = ["post_type" => $this->cpt_type];
    $args[$key] = $value;
    return (new WP_Query($args))->found_posts > 0;
  }


  private function ultimate_ai_get_template_data_boilerplate()
  {
    return [
      "name" => "Unknown tool",
      "description" => "Make your work faster with ai",
      "key" => "unknown_tool",
      "icon_url" => UAI_ASSETS_URL_IMG . "tool_icons/default.svg",
      "is_active" => true,
      "is_featured" => false,
      "input_fields" => [],
      "prompt" => ""
    ];
  }

  public function ultimate_ai_validate_field($field)
  {
    if (empty($field) || !is_array($field)) return false;

    $valid_field_type = ["text", "textarea", "number", "language", "number_of_result", "tone_of_voice", "result_format", "dropdown", "radio_button"];
    $valid_must_fields = ["field_type", "field_name", "key"];
    $is_valid = true;

    foreach ($valid_must_fields as $f) {
      if (empty($field[$f])) {
        $is_valid = false;
      }
      if (!in_array($field["field_type"], $valid_field_type)) {
        $is_valid = false;
      }
    }

    return $is_valid;
  }
}
